
#include "SRAM.h"
#include <plib.h>

//#define USE_BIT_BANG

static void SRAMDMASetup(void) {
    IEC2bits.DMA2IE = 0;           // disable DMA channel 2 interrupts
    IFS2bits.DMA2IF = 0;           // clear any existing DMA channel 2 interrupt flag
    DMACONbits.ON = 1;             // enable DMA unit
    DCH2CON = 0;                   // channel off, no chaining
    DCH2CONbits.CHPRI = 2;         // priority 2
    DCH2ECON = 0;
    DCH2ECONbits.SIRQEN = 1;       // start transfer on interrupt
    DCH2ECONbits.CHSIRQ = 51;      // PMP interrupt
    DCH2CSIZ = 1;                   // bytes transferred per event
    DCH2INTCLR = 0x00ff00ff;       // clear existing events, disable all interrupts
}

static void SRAMPMPSetup(void) {
    PMCONbits.PTRDEN = 1;
    PMCONbits.PTWREN = 1;
    PMMODEbits.WAITE = 0;
    PMMODEbits.WAITM = 2;
    PMMODEbits.WAITB = 0;
    PMMODEbits.MODE = 2;
    PMMODEbits.INCM = 1;
    PMADDR = 0;
    PMAEN = 0xFFFF;
    PMCONbits.ON = 1;

    PMMODEbits.IRQM = 1; // generate interrupt after each read/write cycle
    SRAMDMASetup();
    LATDbits.LATD6 = 0;
}

void SRAMSetup(void) {
    // SRAM CS = RD6
    PORTSetPinsDigitalOut(IOPORT_D, BIT_6);
    LATDbits.LATD6 = 1;
    // PMRD = RD5
    PORTSetPinsDigitalOut(IOPORT_D, BIT_5);
    LATDbits.LATD5 = 1;
    // PMWR = RD4
    PORTSetPinsDigitalOut(IOPORT_D, BIT_4);
    LATDbits.LATD4 = 1;

    // PMA0 = AN15/RB15
    PORTSetPinsDigitalOut(IOPORT_B, BIT_15);
    LATBbits.LATB15 = 0;
    ANSELBbits.ANSB15 = 0;
    // PMA1 = AN14/RB14
    PORTSetPinsDigitalOut(IOPORT_B, BIT_14);
    ANSELBbits.ANSB14 = 0;
    LATBbits.LATB14 = 0;
    // PMA2 = AN19/RG9
    PORTSetPinsDigitalOut(IOPORT_G, BIT_9);
    ANSELGbits.ANSG9 = 0;
    LATGbits.LATG9 = 0;
    // PMA3 = AN18/RG8
    PORTSetPinsDigitalOut(IOPORT_G, BIT_8);
    ANSELGbits.ANSG8 = 0;
    LATGbits.LATG8 = 0;
    // PMA4 = AN17/RG7
    PORTSetPinsDigitalOut(IOPORT_G, BIT_7);
    ANSELGbits.ANSG7 = 0;
    LATGbits.LATG7 = 0;
    // PMA5 = AN16/RG6
    PORTSetPinsDigitalOut(IOPORT_G, BIT_6);
    ANSELGbits.ANSG6 = 0;
    LATGbits.LATG6 = 0;
    // PMA6 = AN0/RB0
    PORTSetPinsDigitalOut(IOPORT_B, BIT_0);
    ANSELBbits.ANSB0 = 0;
    LATBbits.LATB0 = 0;
    // PMA7 = AN9/RB9
    PORTSetPinsDigitalOut(IOPORT_B, BIT_9);
    ANSELBbits.ANSB9 = 0;
    LATBbits.LATB9 = 0;
    // PMA8 = RF5
    PORTSetPinsDigitalOut(IOPORT_F, BIT_5);
    LATFbits.LATF5 = 0;
    // PMA9 = RF4
    PORTSetPinsDigitalOut(IOPORT_F, BIT_4);
    LATFbits.LATF4 = 0;
    // PMA10 = AN13/RB13
    PORTSetPinsDigitalOut(IOPORT_B, BIT_13);
    ANSELBbits.ANSB13 = 0;
    LATBbits.LATB13 = 0;
    // PMA11 = AN12/RB12
    PORTSetPinsDigitalOut(IOPORT_B, BIT_12);
    ANSELBbits.ANSB12 = 0;
    LATBbits.LATB12 = 0;
    // PMA12 = AN11/RB11
    PORTSetPinsDigitalOut(IOPORT_B, BIT_11);
    ANSELBbits.ANSB11 = 0;
    LATBbits.LATB11 = 0;
    // PMA13 = AN10/RB10
    PORTSetPinsDigitalOut(IOPORT_B, BIT_10);
    ANSELBbits.ANSB10 = 0;
    LATBbits.LATB10 = 0;
    // PMA14 (PMCS1) = RD11
    PORTSetPinsDigitalOut(IOPORT_D, BIT_11);
    LATDbits.LATD11 = 0;
    // PMA15 (PMCS2) = RD10
    PORTSetPinsDigitalOut(IOPORT_D, BIT_10);
    LATDbits.LATD10 = 0;

    // PMA16 [GPIO] = AN1/RB1
    PORTSetPinsDigitalOut(IOPORT_B, BIT_1);
    ANSELBbits.ANSB1 = 0;
    LATBbits.LATB1 = 0;
    // PMA17 [GPIO] = AN2/RB2
    PORTSetPinsDigitalOut(IOPORT_B, BIT_2);
    ANSELBbits.ANSB2 = 0;
    LATBbits.LATB2 = 0;
    // PMA18 [GPIO] = AN3/RB3
    PORTSetPinsDigitalOut(IOPORT_B, BIT_3);
    ANSELBbits.ANSB3 = 0;
    LATBbits.LATB3 = 0;
    // PMA19 [GPIO] = AN4/RB4
    PORTSetPinsDigitalOut(IOPORT_B, BIT_4);
    ANSELBbits.ANSB4 = 0;
    LATBbits.LATB4 = 0;

    // PMD0 = RE0
    PORTSetPinsDigitalIn(IOPORT_E, BIT_0);
    // PMD1 = RE1
    PORTSetPinsDigitalIn(IOPORT_E, BIT_1);
    // PMD2 = AN20/RE2
    PORTSetPinsDigitalIn(IOPORT_E, BIT_2);
    ANSELEbits.ANSE2 = 0;
    // PMD3 = RE3
    PORTSetPinsDigitalIn(IOPORT_E, BIT_3);
    // PMD4 = AN21/RE4
    PORTSetPinsDigitalIn(IOPORT_E, BIT_4);
    ANSELEbits.ANSE4 = 0;
    // PMD5 = AN22/RE5
    PORTSetPinsDigitalIn(IOPORT_E, BIT_5);
    ANSELEbits.ANSE5 = 0;
    // PMD6 = AN23/RE6
    PORTSetPinsDigitalIn(IOPORT_E, BIT_6);
    ANSELEbits.ANSE6 = 0;
    // PMD7 = AN27/RE7
    PORTSetPinsDigitalIn(IOPORT_E, BIT_7);
    ANSELEbits.ANSE7 = 0;

#ifndef USE_BIT_BANG
    SRAMPMPSetup();
#endif
}

unsigned char IsSRAMBusy(void) {
    return DCH2CONbits.CHEN;
}

enum { RAM_to_SRAM, SRAM_to_RAM } SRAM_copy_mode;
unsigned long SRAM_copy_bytes_remaining, SRAM_last_block;
void* SRAM_copy_ptr;

static void SRAM_Select_Block(unsigned long block) {
    // PMA16 [GPIO] = AN1/RB1
    LATBbits.LATB1 = block&1;
    block >>= 1;
    // PMA17 [GPIO] = AN2/RB2
    LATBbits.LATB2 = block&1;
    block >>= 1;
    // PMA18 [GPIO] = AN3/RB3
    LATBbits.LATB3 = block&1;
    block >>= 1;
    // PMA19 [GPIO] = AN4/RB4
    LATBbits.LATB4 = block&1;
}

void __ISR(_DMA2_VECTOR,IPL2) __DMA2Interrupt(void) {
    if( SRAM_copy_bytes_remaining ) {
        unsigned long bytes = SRAM_copy_bytes_remaining;
        SRAM_last_block = (SRAM_last_block+1) & 15;
        SRAM_Select_Block(SRAM_last_block);
        PMADDR = 0;
        if( bytes > 0x10000 )
            bytes = 0x10000;
        SRAM_copy_bytes_remaining -= bytes;
        if( SRAM_copy_mode = RAM_to_SRAM ) {
            DCH2SSA = VirtToPhys(SRAM_copy_ptr);
            SRAM_copy_ptr = ((char*)SRAM_copy_ptr) + bytes;
            DCH2SSIZ = bytes;
            DCH2CONbits.CHEN = 1;           // turn channel on
            DCH2ECONbits.CFORCE = 1;        // initiate transfer
        } else {
            DCH2DSA = VirtToPhys(SRAM_copy_ptr);
            SRAM_copy_ptr = ((char*)SRAM_copy_ptr) + bytes;
            DCH2SSIZ = bytes;
            DCH2CONbits.CHEN = 1;           // turn channel on
            PMDIN;                          // initiate transfer
        }
    } else {
        IEC2bits.DMA2IE = 0;
    }
    IFS2bits.DMA2IF = 0;
}

//unsigned long block, base, bytes;
void SRAMtoRAMCopy(unsigned long srcaddr, void* dest, unsigned long numbytes) {
#ifdef USE_BIT_BANG
    while( numbytes ) {
        ((unsigned char*)dest)[0] = SRAMReadByte(srcaddr, 0);
        dest = ((unsigned char*)dest) + 1;
        ++srcaddr;
        --numbytes;
    }
#else
    unsigned long block = srcaddr >> 16;
    unsigned long base = srcaddr & 0xFFFF;
    unsigned long bytes = numbytes;
    if( base + bytes > 0x10000 )
        bytes = 0x10000 - base;

    // wait for last copy to complete (if still in progress)
    while( DCH2CONbits.CHEN )
        ;

    // set up start address
    SRAM_Select_Block(block);
    PMADDR = base;
    PMDIN;

    DCH2SSA = VirtToPhys(&PMDIN);   // transfer source physical address
    DCH2DSA = VirtToPhys(dest);     // transfer destination physical address
    DCH2SSIZ = 1;                   // source size
    DCH2DSIZ = numbytes;            // destination size
    if( bytes < numbytes ) {
        SRAM_copy_mode = SRAM_to_RAM;
        SRAM_last_block = block;
        SRAM_copy_bytes_remaining = numbytes - bytes;
        SRAM_copy_ptr = (void*)((char*)dest + bytes);
        DCH2INTbits.CHBCIE = 1;     // enable interrupt on block copy complete
        IFS2bits.DMA2IF = 0;
        IEC2bits.DMA2IE = 1;
    } else {
        DCH2INTbits.CHBCIE = 0;     // no interrupt on block copy complete
    }
    DCH2CONbits.CHEN = 1;           // turn channel on
//    if( !DCH2CONbits.CHBUSY )
//      DCH2ECONbits.CFORCE = 1;        // initiate transfer
#endif
}

void RAMtoSRAMCopy(const void* src, unsigned long destaddr, unsigned long numbytes) {
#ifdef USE_BIT_BANG
    while( numbytes ) {
        SRAMWriteByte(destaddr, ((const unsigned char*)src)[0]);
        src  = ((const unsigned char*)src) + 1;
        ++destaddr;
        --numbytes;
    }
#else
    unsigned long block = destaddr >> 16;
    unsigned long base = destaddr & 0xFFFF;
    unsigned long bytes = numbytes;
    if( base + bytes > 0x10000 )
        bytes = 0x10000 - base;

    // wait for last copy to complete (if still in progress)
    while( DCH2CONbits.CHEN )
        ;

    // set up start address
    SRAM_Select_Block(block);
    PMADDR = base;

    DCH2SSA = VirtToPhys(src);      // transfer source physical address
    DCH2DSA = VirtToPhys(&PMDIN);   // transfer destination physical address
    DCH2SSIZ = numbytes;            // source size
    DCH2DSIZ = 1;                   // destination size
    if( bytes < numbytes ) {
        SRAM_copy_mode = RAM_to_SRAM;
        SRAM_last_block = block;
        SRAM_copy_bytes_remaining = numbytes - bytes;
        SRAM_copy_ptr = (void*)((const char*)src + bytes);
        DCH2INTbits.CHBCIE = 1;     // enable interrupt on block copy complete
        IFS2bits.DMA2IF = 0;
        IEC2bits.DMA2IE = 1;
    } else {
        DCH2INTbits.CHBCIE = 0;     // no interrupt on block copy complete
    }
    DCH2CONbits.CHEN = 1;           // turn channel on
    DCH2ECONbits.CFORCE = 1;        // initiate transfer
#endif
}

unsigned char TestSRAM(unsigned char* buf, unsigned long size) {
    unsigned int i, j;
    CNPDE |= 0xFF;
    for( i = 0; i < size/256; ++i ) {
        while( IsSRAMBusy() )
            ;
        for( j = 0; j < 256; ++j )
            buf[j] = (i+j)&255;
        RAMtoSRAMCopy(buf, i * 256, 256);
    }
    while( IsSRAMBusy() )
        ;
    memset(buf, 0, 256);
    for( i = 0; i < size/256; ++i ) {
        memset(buf, 0xCC, 256);
        SRAMtoRAMCopy(i * 256, buf, 256);
        while( IsSRAMBusy() )
            ;
        for( j = 0; j < 256; ++j ) {
            if( buf[j] != (unsigned char)((i+j)&255) ) {
                CNPDE &= ~0xFF;
                return 1;
            }
        }
    }
    CNPDE &= ~0xFF;
    return 0;
}

void EraseSRAM(unsigned char* buf, unsigned long size) {
    unsigned int i;
    memset(buf, 0, 256);
    for( i = 0; i < size/256; ++i )
        RAMtoSRAMCopy(buf, i * 256, 256);
    while( IsSRAMBusy() )
        ;
}

static void SRAMSetAddress(unsigned long Address) {
    // PMA0 = AN15/RB15
    LATBbits.LATB15 = Address&1;
    Address >>= 1;
    // PMA1 = AN14/RB14
    LATBbits.LATB14 = Address&1;
    Address >>= 1;
    // PMA2 = AN19/RG9
    LATGbits.LATG9 = Address&1;
    Address >>= 1;
    // PMA3 = AN18/RG8
    LATGbits.LATG8 = Address&1;
    Address >>= 1;
    // PMA4 = AN17/RG7
    LATGbits.LATG7 = Address&1;
    Address >>= 1;
    // PMA5 = AN16/RG6
    LATGbits.LATG6 = Address&1;
    Address >>= 1;
    // PMA6 = AN0/RB0
    LATBbits.LATB0 = Address&1;
    Address >>= 1;
    // PMA7 = AN9/RB9
    LATBbits.LATB9 = Address&1;
    Address >>= 1;
    // PMA8 = RF5
    LATFbits.LATF5 = Address&1;
    Address >>= 1;
    // PMA9 = RF4
    LATFbits.LATF4 = Address&1;
    Address >>= 1;
    // PMA10 = AN13/RB13
    LATBbits.LATB13 = Address&1;
    Address >>= 1;
    // PMA11 = AN12/RB12
    LATBbits.LATB12 = Address&1;
    Address >>= 1;
    // PMA12 = AN11/RB11
    LATBbits.LATB11 = Address&1;
    Address >>= 1;
    // PMA13 = AN10/RB10
    LATBbits.LATB10 = Address&1;
    Address >>= 1;
    // PMA14 (PMCS1) = RD11
    LATDbits.LATD11 = Address&1;
    Address >>= 1;
    // PMA15 (PMCS2) = RD10
    LATDbits.LATD10 = Address&1;
    Address >>= 1;

    // PMA16 [GPIO] = AN1/RB1
    LATBbits.LATB1 = Address&1;
    Address >>= 1;
    // PMA17 [GPIO] = AN2/RB2
    LATBbits.LATB2 = Address&1;
    Address >>= 1;
    // PMA18 [GPIO] = AN3/RB3
    LATBbits.LATB3 = Address&1;
    Address >>= 1;
    // PMA19 [GPIO] = AN4/RB4
    LATBbits.LATB4 = Address&1;
}

unsigned char SRAMReadByte(unsigned long Address, unsigned char pulldown) {
    unsigned char ret;

    if( pulldown )
        CNPDE |= 0xFF;
    else
        CNPDE &= ~0xFF;

    SRAMSetAddress(Address);
    Nop();
    Nop();
    // SRAM CS = RD6
    LATDbits.LATD6 = 0;
    // PMRD = RD5
    LATDbits.LATD5 = 0;
    Nop();
    Nop();
    Nop();
    Nop();
    ret = PORTE&0xFF;
    // PMRD = RD5
    LATDbits.LATD5 = 1;
    // SRAM CS = RD6
    LATDbits.LATD6 = 1;

    if( pulldown )
        CNPDE &= ~0xFF;

    return ret;
}

void SRAMWriteByte(unsigned long Address, unsigned char data) {
    SRAMSetAddress(Address);
    Nop();
    Nop();
    TRISE &= ~0xFF;
    LATE = (LATE&~0xFF)|data;
    // SRAM CS = RD6
    Nop();
    Nop();
    LATDbits.LATD6 = 0;
    // PMWR = RD4
    LATDbits.LATD4 = 0;
    Nop();
    Nop();
    Nop();
    Nop();
    // PMWR = RD4
    LATDbits.LATD4 = 1;
    // SRAM CS = RD6
    LATDbits.LATD6 = 1;
    Nop();
    Nop();

    TRISE |= 0xFF;
    LATE &= ~0xFF;
}
